class ToastUINotesEditor {
    constructor() {
        this.editor = null;
        this.statusElement = document.getElementById('status');
        this.wordCountElement = document.getElementById('word-count');
        this.saveTimeout = null;
        this.isDarkTheme = false;
        this.isChatOpen = false; // Add chat state tracking
        this.chatHistory = []; // Store conversation history
        this.openaiApiKey = null; // Store API key
        this.selectedModel = 'gpt-4o-mini'; // Default model
        
        this.init();
    }

    init() {
        this.loadTheme();
        this.initToastUIEditor();
        this.setupEventListeners();
        this.loadContent();
        this.updateWordCount();
    }

    createCustomButton(id, text, tooltip) {
        const button = document.createElement('button');
        button.type = 'button';
        button.className = 'custom-toolbar-button';
        button.style.cssText = `
            background: transparent;
            border: 1px solid transparent;
            padding: 6px 8px;
            margin: 0 2px;
            cursor: pointer;
            border-radius: 3px;
            font-size: 14px;
            font-weight: normal;
            color: #666;
            min-width: 24px;
            height: 24px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            position: relative;
            z-index: 1000;
        `;
        button.title = tooltip;
        button.textContent = text;
        button.tabIndex = 0;
        
        // Add hover effect
        button.addEventListener('mouseenter', () => {
            button.style.backgroundColor = '#f5f5f5';
            button.style.borderColor = '#ddd';
        });
        button.addEventListener('mouseleave', () => {
            button.style.backgroundColor = 'transparent';
            button.style.borderColor = 'transparent';
        });
        
        // Add focus effect
        button.addEventListener('focus', () => {
            button.style.backgroundColor = '#f5f5f5';
            button.style.borderColor = '#007bff';
            button.style.outline = 'none';
        });
        button.addEventListener('blur', () => {
            button.style.backgroundColor = 'transparent';
            button.style.borderColor = 'transparent';
        });
        
        if (id === 'export') {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                console.log('Export button clicked!');
                this.exportToMarkdown();
            });
        } else if (id === 'theme') {
            button.id = 'theme-toggle-toolbar';
            button.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                console.log('Theme button clicked!');
                this.toggleTheme();
            });
        }
        
        return button;
    }

    createCustomStatusButton(id, text, tooltip) {
        const button = document.createElement('button');
        button.type = 'button';
        button.className = 'custom-status-button';
        button.title = tooltip;
        button.textContent = text;
        button.tabIndex = 0;
        
        if (id === 'export') {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                console.log('Export button clicked!');
                this.exportToMarkdown();
            });
        } else if (id === 'theme') {
            button.id = 'theme-toggle-status';
            button.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                console.log('Theme button clicked!');
                this.toggleTheme();
            });
        } else if (id === 'chat') {
            button.id = 'chat-toggle-status';
            button.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                console.log('Chat button clicked!');
                this.toggleChat();
            });
        }
        
        return button;
    }

    addCustomStatusButtons() {
        const statusBar = document.querySelector('.status-bar');
        
        if (statusBar) {
            // Check if we already added buttons to avoid duplicates
            if (statusBar.querySelector('.custom-status-button')) {
                console.log('Custom status buttons already exist, skipping...');
                return;
            }
            
            // Create a container for the right side buttons
            const rightContainer = document.createElement('div');
            rightContainer.style.cssText = `
                display: flex;
                align-items: center;
                gap: 10px;
                margin-left: auto;
            `;
            
            // Add export button
            const exportBtn = this.createCustomStatusButton('export', '↓ Export', 'Export as Markdown');
            rightContainer.appendChild(exportBtn);
            
            // Add separator
            const separator1 = document.createElement('span');
            separator1.textContent = '|';
            separator1.style.cssText = 'color: #ccc; font-size: 12px;';
            rightContainer.appendChild(separator1);
            
            // Add chat button
            const chatBtn = this.createCustomStatusButton('chat', '💬 Chat', 'Toggle Chat (Cmd+/) - Highlight text to add to chat');
            rightContainer.appendChild(chatBtn);
            
            // Add separator
            const separator2 = document.createElement('span');
            separator2.textContent = '|';
            separator2.style.cssText = 'color: #ccc; font-size: 12px;';
            rightContainer.appendChild(separator2);
            
            // Add theme button  
            const themeBtn = this.createCustomStatusButton('theme', this.isDarkTheme ? '☀ Light' : '🌙 Dark', 'Toggle Theme');
            rightContainer.appendChild(themeBtn);
            
            // Add the container to status bar (it will align right due to margin-left: auto)
            statusBar.appendChild(rightContainer);
            
            console.log('Custom status buttons added successfully!');
        } else {
            console.log('Could not find status bar to add custom buttons');
        }
    }

    initToastUIEditor() {
        const { Editor } = toastui;
        
        this.editor = new Editor({
            el: document.querySelector('#editor'),
            height: '100%',
            initialEditType: 'markdown',
            previewStyle: 'none',
            viewer: false,
            preview: false,  // Explicitly disable preview
            placeholder: 'Start typing your notes here...\n\nUse markdown syntax:\n# Heading 1\n## Heading 2\n- List item\n- [ ] Checkbox\n> Quote\n`code`\n```\ncode block\n```',
            theme: this.isDarkTheme ? 'dark' : 'light',
            toolbarItems: [
                ['heading', 'bold', 'italic', 'strike'],
                ['hr', 'quote'],
                ['ul', 'ol', 'task', 'indent', 'outdent'],
                ['table', 'image', 'link'],
                ['code', 'codeblock'],
            ],
            hooks: {
                addImageBlobHook: (blob, callback) => {
                    // Handle image uploads if needed
                    const reader = new FileReader();
                    reader.onload = (e) => {
                        callback(e.target.result, 'image');
                    };
                    reader.readAsDataURL(blob);
                }
            },
            usageStatistics: false,
            hideModeSwitch: true,
            linkAttributes: {
                target: '_blank',
                rel: 'noopener noreferrer'
            }
        });

        // Listen for content changes
        this.editor.on('change', () => {
            this.updateWordCount();
            this.autoSave();
        });

        // Force the editor to stay in markdown mode
        this.enforceMarkdownMode();

        // Fix for webkit paste scrolling bug - preserve scroll position on paste
        this.setupPasteScrollFix();

        // Add custom buttons to status bar and update theme button after editor is ready
        setTimeout(() => {
            this.addCustomStatusButtons();
            this.updateThemeButton();
        }, 500); // Increased delay to ensure editor is fully loaded
    }

    enforceMarkdownMode() {
        // Ensure editor stays in markdown mode
        setTimeout(() => {
            try {
                // Force markdown mode if available
                if (this.editor && typeof this.editor.changeMode === 'function') {
                    this.editor.changeMode('markdown');
                }
                
                // Hide any preview elements that might appear
                this.hidePreviewElements();
                
                // Monitor for mode changes and force back to markdown
                if (this.editor && this.editor.on) {
                    this.editor.on('changeMode', (mode) => {
                        if (mode !== 'markdown') {
                            console.log('Forcing editor back to markdown mode from:', mode);
                            setTimeout(() => {
                                this.editor.changeMode('markdown');
                                this.hidePreviewElements();
                            }, 0);
                        }
                    });
                }
                
                // Set up periodic checker to hide preview elements
                this.startPreviewHider();
                
            } catch (error) {
                console.warn('Could not enforce markdown mode:', error);
            }
        }, 1000);
    }

    hidePreviewElements() {
        const editorElement = document.querySelector('#editor');
        if (editorElement) {
            // Comprehensive list of preview-related selectors including splitters and dividers
            const previewSelectors = [
                '.toastui-editor-md-preview',
                '.toastui-editor-md-preview-container', 
                '.toastui-editor-preview',
                '.toastui-editor-preview-container',
                '.toastui-editor-md-splitter',
                '.toastui-editor-md-vertical-splitter',
                '.toastui-editor-md-splitter-bar',
                '.toastui-editor-split-divider',
                '[data-language="preview"]',
                '.toastui-editor-contents[data-language="preview"]',
                '.toastui-editor-md-preview-style',
                '.toastui-editor-defaultUI .toastui-editor-md-preview',
                '.toastui-editor-defaultUI .toastui-editor-preview'
            ];
            
            let hiddenCount = 0;
            
            previewSelectors.forEach(selector => {
                const elements = editorElement.querySelectorAll(selector);
                elements.forEach(el => {
                    if (el.style.display !== 'none' || el.offsetHeight > 0 || el.offsetWidth > 0) {
                        console.log('Hiding preview element:', {
                            selector: selector,
                            className: el.className,
                            offsetHeight: el.offsetHeight,
                            offsetWidth: el.offsetWidth,
                            element: el
                        });
                        
                        // Nuclear option - completely remove from layout
                        el.style.cssText = `
                            display: none !important;
                            width: 0 !important;
                            height: 0 !important;
                            min-width: 0 !important;
                            min-height: 0 !important;
                            max-width: 0 !important;
                            max-height: 0 !important;
                            margin: 0 !important;
                            padding: 0 !important;
                            border: none !important;
                            overflow: hidden !important;
                            position: absolute !important;
                            left: -9999px !important;
                            top: -9999px !important;
                            visibility: hidden !important;
                            opacity: 0 !important;
                        `;
                        
                        hiddenCount++;
                    }
                });
            });
            
            // Additional check: look for any elements with specific heights that might be preview remnants
            const allElements = editorElement.querySelectorAll('*');
            allElements.forEach(el => {
                const computedStyle = window.getComputedStyle(el);
                const height = el.offsetHeight;
                
                // Look for suspiciously positioned elements at the bottom
                if (height > 0 && height < 50 && (
                    el.className.includes('preview') ||
                    el.className.includes('splitter') ||
                    el.className.includes('divider') ||
                    computedStyle.position === 'absolute' && computedStyle.bottom === '0px'
                )) {
                    console.log('Found suspicious bottom element:', {
                        className: el.className,
                        height: height,
                        bottom: computedStyle.bottom,
                        position: computedStyle.position,
                        element: el
                    });
                    
                    // Hide it
                    el.style.cssText = `
                        display: none !important;
                        height: 0 !important;
                        width: 0 !important;
                        position: absolute !important;
                        left: -9999px !important;
                        top: -9999px !important;
                    `;
                    hiddenCount++;
                }
            });
            
            if (hiddenCount > 0) {
                console.log(`Hidden ${hiddenCount} preview/splitter elements`);
            }
        }
    }

    startPreviewHider() {
        // Check every 500ms for any preview elements and hide them
        setInterval(() => {
            this.hidePreviewElements();
        }, 500);
        
        // Also use MutationObserver for immediate response to DOM changes
        const editorElement = document.querySelector('#editor');
        if (editorElement) {
            const observer = new MutationObserver((mutations) => {
                mutations.forEach((mutation) => {
                    if (mutation.type === 'childList') {
                        mutation.addedNodes.forEach((node) => {
                            if (node.nodeType === Node.ELEMENT_NODE) {
                                // Check if the added node is a preview element
                                const element = node;
                                if (element.classList && (
                                    element.classList.contains('toastui-editor-md-preview') ||
                                    element.classList.contains('toastui-editor-preview') ||
                                    element.classList.contains('toastui-editor-md-preview-container') ||
                                    element.classList.contains('toastui-editor-preview-container') ||
                                    element.classList.contains('toastui-editor-md-splitter')
                                )) {
                                    console.log('MutationObserver: Immediately hiding preview element:', element.className);
                                    element.style.display = 'none !important';
                                    element.style.width = '0';
                                    element.style.height = '0';
                                    element.style.overflow = 'hidden';
                                }
                                
                                // Also check for preview elements within the added node
                                const previewChildren = element.querySelectorAll(
                                    '.toastui-editor-md-preview, .toastui-editor-preview, .toastui-editor-md-preview-container, .toastui-editor-preview-container, .toastui-editor-md-splitter'
                                );
                                previewChildren.forEach(child => {
                                    console.log('MutationObserver: Hiding child preview element:', child.className);
                                    child.style.display = 'none !important';
                                    child.style.width = '0';
                                    child.style.height = '0';
                                    child.style.overflow = 'hidden';
                                });
                            }
                        });
                    }
                });
            });
            
            observer.observe(editorElement, {
                childList: true,
                subtree: true
            });
        }
        
        // Additional protection: Monitor for any layout changes that might indicate preview mode
        this.monitorEditorLayout();
    }

    monitorEditorLayout() {
        setInterval(() => {
            try {
                const editorElement = document.querySelector('#editor');
                if (editorElement) {
                    // Ensure markdown container always takes full width
                    const mdContainer = editorElement.querySelector('.toastui-editor-md-container');
                    if (mdContainer) {
                        mdContainer.style.width = '100%';
                        mdContainer.style.flex = '1';
                        mdContainer.style.height = '100%';
                        
                        // Check if there are multiple containers (indicating split view)
                        const containers = editorElement.querySelectorAll('.toastui-editor-md-container, .toastui-editor-preview-container');
                        if (containers.length > 1) {
                            console.log('Detected multiple containers, hiding non-markdown containers');
                            containers.forEach((container, index) => {
                                if (index > 0 || container.classList.contains('toastui-editor-preview-container')) {
                                    container.style.display = 'none';
                                    container.style.width = '0';
                                    container.style.height = '0';
                                }
                            });
                        }
                    }
                    
                    // Check for any bottom spacing issues
                    this.eliminateBottomSpace();
                    
                    // Force editor mode if it's not markdown
                    if (this.editor && typeof this.editor.getCurrentModeEditor === 'function') {
                        const currentMode = this.editor.getCurrentModeEditor();
                        if (currentMode && currentMode.mode !== 'markdown') {
                            console.log('Current mode is not markdown, forcing change');
                            this.editor.changeMode('markdown');
                        }
                    }
                }
            } catch (error) {
                // Silently handle errors to avoid spam
            }
        }, 1000); // Check every second
    }

    eliminateBottomSpace() {
        const editorElement = document.querySelector('#editor');
        if (editorElement) {
            const editorRect = editorElement.getBoundingClientRect();
            const editorBottom = editorRect.bottom;
            
            // Look for any elements positioned near the bottom that might be taking up space
            const allElements = editorElement.querySelectorAll('*');
            allElements.forEach(el => {
                const rect = el.getBoundingClientRect();
                const height = rect.height;
                
                // If element is near the bottom and has suspicious height (2%-5% of screen)
                if (height > 0 && height < window.innerHeight * 0.1 && 
                    rect.bottom > editorBottom - height - 10) {
                    
                    const computedStyle = window.getComputedStyle(el);
                    
                    // Check if it looks like a preview remnant
                    if (el.className.includes('preview') || 
                        el.className.includes('splitter') ||
                        el.className.includes('container') ||
                        computedStyle.borderTop || 
                        computedStyle.borderBottom) {
                        
                        console.log('Eliminating bottom space element:', {
                            className: el.className,
                            height: height,
                            position: rect.top + ',' + rect.bottom,
                            element: el
                        });
                        
                        // Nuclear removal
                        el.style.cssText = `
                            display: none !important;
                            height: 0 !important;
                            width: 0 !important;
                            margin: 0 !important;
                            padding: 0 !important;
                            border: none !important;
                            position: absolute !important;
                            left: -9999px !important;
                            top: -9999px !important;
                        `;
                    }
                }
            });
            
            // Force the main toastui-editor container to take full height
            const toastuiEditor = editorElement.querySelector('.toastui-editor');
            if (toastuiEditor) {
                toastuiEditor.style.height = '100%';
                toastuiEditor.style.display = 'flex';
                toastuiEditor.style.flexDirection = 'column';
            }
        }
    }

    setupPasteScrollFix() {
        // Simplified paste scroll fix - less intrusive
        setTimeout(() => {
            const editorElement = document.querySelector('#editor');
            const codeMirrorElement = editorElement?.querySelector('.CodeMirror');
            
            if (codeMirrorElement) {
                console.log('Setting up simplified paste scroll fix with preview elimination');
                
                let savedScrollTop = 0;
                
                // Enhanced paste handler
                codeMirrorElement.addEventListener('paste', () => {
                    try {
                        const scrollInfo = this.editor.getCodeMirror().getScrollInfo();
                        savedScrollTop = scrollInfo.top;
                        
                        // Immediately hide any preview elements that might appear
                        this.hidePreviewElements();
                        this.eliminateBottomSpace();
                        
                        // Restore after a short delay
                        setTimeout(() => {
                            // Hide preview elements again after paste is processed
                            this.hidePreviewElements();
                            this.eliminateBottomSpace();
                            
                            const currentScrollInfo = this.editor.getCodeMirror().getScrollInfo();
                            // Only restore if scroll moved significantly
                            if (Math.abs(currentScrollInfo.top - savedScrollTop) > 100) {
                                this.editor.getCodeMirror().scrollTo(0, savedScrollTop);
                            }
                        }, 200);
                        
                        // Additional cleanup after a longer delay
                        setTimeout(() => {
                            this.hidePreviewElements();
                            this.eliminateBottomSpace();
                        }, 500);
                        
                    } catch (error) {
                        console.warn('Paste scroll fix error:', error);
                    }
                });
            }
        }, 1000);
    }

    setupEventListeners() {
        // Listen for storage changes (real-time sync across tabs)
        chrome.storage.onChanged.addListener((changes, namespace) => {
            if (namespace === 'local' && changes['newtab-notes-content']) {
                this.handleStorageChange(changes['newtab-notes-content']);
            }
        });

        // Listen for theme changes
        chrome.storage.onChanged.addListener((changes, namespace) => {
            if (namespace === 'local' && changes['newtab-notes-theme']) {
                this.isDarkTheme = changes['newtab-notes-theme'].newValue === 'dark';
                this.applyTheme();
                this.updateThemeButton();
            }
        });

        // Keyboard shortcuts
        document.addEventListener('keydown', (e) => {
            // Cmd/Ctrl + / to toggle chat
            if ((e.metaKey || e.ctrlKey) && e.key === '/') {
                e.preventDefault();
                this.toggleChat();
            }
            
            // Shift + Ctrl/Cmd + R to manually scroll to top of editor (in case of scroll issues)
            if (e.key === 'r' || e.key === 'R') {
                if ((e.metaKey || e.ctrlKey) && e.shiftKey) {
                    e.preventDefault();
                    try {
                        this.editor.getCodeMirror().scrollTo(0, 0);
                        console.log('Editor scrolled to top');
                    } catch (error) {
                        console.warn('Could not scroll to top:', error);
                    }
                }
            }
            
            // Shift + Ctrl/Cmd + H to manually hide all preview elements
            if (e.key === 'h' || e.key === 'H') {
                if ((e.metaKey || e.ctrlKey) && e.shiftKey) {
                    e.preventDefault();
                    console.log('Manual preview cleanup triggered');
                    this.hidePreviewElements();
                    this.eliminateBottomSpace();
                    
                    // Force markdown mode
                    if (this.editor && typeof this.editor.changeMode === 'function') {
                        this.editor.changeMode('markdown');
                    }
                    
                    console.log('Preview cleanup complete');
                }
            }
        });
    }

    autoSave() {
        if (this.saveTimeout) {
            clearTimeout(this.saveTimeout);
        }
        
        this.saveTimeout = setTimeout(() => {
            this.saveContent();
        }, 1000);
        
        this.updateStatus('Saving...');
    }

    saveContent() {
        const content = this.editor.getMarkdown();
        chrome.storage.local.set({ 'newtab-notes-content': content }, () => {
            this.updateStatus('Saved');
            setTimeout(() => {
                this.updateStatus('Ready');
            }, 2000);
        });
    }

    loadContent() {
        chrome.storage.local.get(['newtab-notes-content'], (result) => {
            if (result['newtab-notes-content'] && this.editor) {
                this.editor.setMarkdown(result['newtab-notes-content']);
            }
        });
    }

    loadTheme() {
        chrome.storage.local.get(['newtab-notes-theme'], (result) => {
            this.isDarkTheme = result['newtab-notes-theme'] === 'dark';
            this.applyTheme();
        });
    }

    toggleTheme() {
        this.isDarkTheme = !this.isDarkTheme;
        this.applyTheme();
        this.updateThemeButton();
        chrome.storage.local.set({ 'newtab-notes-theme': this.isDarkTheme ? 'dark' : 'light' });
    }

    updateThemeButton() {
        // Update the status bar theme button
        const statusThemeBtn = document.getElementById('theme-toggle-status');
        
        if (statusThemeBtn) {
            statusThemeBtn.textContent = this.isDarkTheme ? '☀ Light' : '🌙 Dark';
        }
    }

    applyTheme() {
        const body = document.body;
        const editorEl = document.querySelector('#editor');
        
        if (this.isDarkTheme) {
            body.classList.add('dark-theme');
            if (editorEl) {
                editorEl.classList.add('toastui-editor-dark');
            }
        } else {
            body.classList.remove('dark-theme');
            if (editorEl) {
                editorEl.classList.remove('toastui-editor-dark');
            }
        }
    }

    updateStatus(message) {
        this.statusElement.textContent = message;
    }

    updateWordCount() {
        if (this.editor) {
            const text = this.editor.getMarkdown();
            const words = text.trim() ? text.trim().split(/\s+/).length : 0;
            this.wordCountElement.textContent = `${words} words`;
        }
    }

    exportToMarkdown() {
        const markdown = this.editor.getMarkdown();
        const blob = new Blob([markdown], { type: 'text/markdown' });
        const url = URL.createObjectURL(blob);
        
        const a = document.createElement('a');
        a.href = url;
        a.download = `notes-${new Date().toISOString().split('T')[0]}.md`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
        
        this.updateStatus('Exported to Markdown');
        setTimeout(() => {
            this.updateStatus('Ready');
        }, 2000);
    }

    handleStorageChange(change) {
        // Only update if the content actually changed and we're not currently editing
        if (change.newValue && change.newValue !== this.editor.getMarkdown()) {
            // Store current cursor position if possible
            try {
                this.editor.setMarkdown(change.newValue, false);
            } catch (e) {
                console.warn('Could not sync content:', e);
            }
        }
    }

    toggleChat() {
        // Get selected text before toggling (in case we're appending)
        const selectedText = this.getSelectedText();
        
        // If chat is already open and we have selected text, append it instead of closing
        if (this.isChatOpen && selectedText) {
            this.appendTextToChat(selectedText);
            return;
        }
        
        // Otherwise, toggle chat open/closed
        this.isChatOpen = !this.isChatOpen;
        this.applyChatLayout();
        this.updateChatButton();
        
        // If opening chat and we have selected text, put it in the input
        if (selectedText && this.isChatOpen) {
            setTimeout(() => {
                const chatInput = document.getElementById('chat-input');
                if (chatInput) {
                    chatInput.value = selectedText;
                    chatInput.focus();
                    chatInput.setSelectionRange(selectedText.length, selectedText.length);
                    // Auto-resize to fit the content
                    this.autoResizeTextarea(chatInput);
                }
            }, 200);
        }
        
        console.log('Chat toggled:', this.isChatOpen ? 'opened' : 'closed');
        if (selectedText) {
            console.log('Selected text:', selectedText);
        }
    }

    getSelectedText() {
        let selectedText = '';
        
        try {
            // Method 1: Try to get from window selection first (most reliable for DOM)
            const selection = window.getSelection();
            if (selection.rangeCount > 0) {
                const range = selection.getRangeAt(0);
                const editorElement = document.querySelector('#editor');
                
                if (editorElement && editorElement.contains(range.commonAncestorContainer)) {
                    selectedText = selection.toString().trim();
                    if (selectedText) {
                        console.log('Got text from DOM selection:', selectedText);
                        return selectedText;
                    }
                }
            }
            
            // Method 2: Try to access CodeMirror directly
            if (this.editor) {
                try {
                    // Look for CodeMirror instance in the DOM
                    const codeMirrorElement = document.querySelector('#editor .CodeMirror');
                    if (codeMirrorElement && codeMirrorElement.CodeMirror) {
                        const cm = codeMirrorElement.CodeMirror;
                        if (cm.getSelection) {
                            selectedText = cm.getSelection().trim();
                            if (selectedText) {
                                console.log('Got text from CodeMirror:', selectedText);
                                return selectedText;
                            }
                        }
                    }
                } catch (e) {
                    console.log('CodeMirror method failed:', e);
                }
                
                // Method 3: Try TOAST UI's getCurrentModeEditor
                try {
                    const currentEditor = this.editor.getCurrentModeEditor();
                    if (currentEditor) {
                        // For markdown mode (CodeMirror-based)
                        if (currentEditor.getSelection) {
                            const selection = currentEditor.getSelection();
                            if (selection && typeof selection === 'string') {
                                selectedText = selection.trim();
                                if (selectedText) {
                                    console.log('Got text from TOAST UI getCurrentModeEditor:', selectedText);
                                    return selectedText;
                                }
                            }
                        }
                        
                        // For WYSIWYG mode
                        if (currentEditor.getSelectedText) {
                            selectedText = currentEditor.getSelectedText().trim();
                            if (selectedText) {
                                console.log('Got text from WYSIWYG mode:', selectedText);
                                return selectedText;
                            }
                        }
                    }
                } catch (e) {
                    console.log('TOAST UI getCurrentModeEditor failed:', e);
                }
                
                // Method 4: Try to access the editor's internal CodeMirror
                try {
                    if (this.editor.mdEditor && this.editor.mdEditor.cm) {
                        const cm = this.editor.mdEditor.cm;
                        selectedText = cm.getSelection().trim();
                        if (selectedText) {
                            console.log('Got text from internal CodeMirror:', selectedText);
                            return selectedText;
                        }
                    }
                } catch (e) {
                    console.log('Internal CodeMirror access failed:', e);
                }
            }
            
        } catch (error) {
            console.log('Error getting selected text:', error);
        }
        
        console.log('No selected text found');
        return selectedText;
    }

    autoResizeTextarea(textarea) {
        if (!textarea) return;
        
        // Reset height to auto to get the correct scrollHeight
        textarea.style.height = 'auto';
        
        // Calculate the new height
        const minHeight = 44; // Single line height with padding
        const maxHeight = 200; // About 10 lines
        const scrollHeight = textarea.scrollHeight;
        
        // Set height between min and max
        const newHeight = Math.min(Math.max(scrollHeight, minHeight), maxHeight);
        textarea.style.height = newHeight + 'px';
        
        // Enable/disable scrolling based on content
        if (scrollHeight > maxHeight) {
            textarea.style.overflowY = 'auto';
        } else {
            textarea.style.overflowY = 'hidden';
        }
    }

    appendTextToChat(text) {
        const chatInput = document.getElementById('chat-input');
        if (chatInput) {
            const currentValue = chatInput.value;
            const separator = currentValue ? '\n\n' : '';
            chatInput.value = currentValue + separator + text;
            chatInput.focus();
            // Move cursor to end
            chatInput.setSelectionRange(chatInput.value.length, chatInput.value.length);
            // Auto-resize after adding text
            this.autoResizeTextarea(chatInput);
            console.log('Appended text to chat:', text);
        }
    }

    applyChatLayout() {
        const container = document.querySelector('.container');
        const editorContainer = document.querySelector('.editor-container');
        
        if (this.isChatOpen) {
            // Create split content wrapper if it doesn't exist
            let splitWrapper = document.querySelector('.split-content-wrapper');
            if (!splitWrapper) {
                splitWrapper = document.createElement('div');
                splitWrapper.className = 'split-content-wrapper';
                
                // Move editor container into the split wrapper
                container.insertBefore(splitWrapper, editorContainer);
                splitWrapper.appendChild(editorContainer);
            }
            
            // Create chat container if it doesn't exist
            let chatContainer = document.querySelector('.chat-container');
            if (!chatContainer) {
                chatContainer = document.createElement('div');
                chatContainer.className = 'chat-container';
                chatContainer.innerHTML = this.createChatHTML();
                splitWrapper.appendChild(chatContainer);
                
                // Initialize chat functionality
                this.initializeChat();
            }
            
            // Apply split layout
            container.classList.add('split-layout');
        } else {
            // Remove split layout
            container.classList.remove('split-layout');
            
            // Move editor back to container and remove split wrapper
            const splitWrapper = document.querySelector('.split-content-wrapper');
            if (splitWrapper) {
                container.insertBefore(editorContainer, splitWrapper);
                splitWrapper.remove();
            }
        }
    }

    createChatHTML() {
        return `
            <div class="chat-header">
                <div class="chat-header-left">
                    <h3>AI Assistant</h3>
                    <span class="chat-status" id="chat-status">Ready to help!</span>
                </div>
                <div class="chat-header-right">
                    <select id="model-selector" class="model-selector" title="Select AI Model">
                        <option value="gpt-4o-mini">GPT-4o Mini (Fast & Affordable)</option>
                        <option value="gpt-4o">GPT-4o (Most Capable)</option>
                    </select>
                    <button id="clear-chat-btn" class="clear-chat-btn" title="Clear Chat History">🗑️</button>
                    <button id="api-key-btn" class="api-key-btn" title="Configure API Key">⚙️</button>
                </div>
            </div>
            <div class="chat-messages" id="chat-messages">
                <div class="message bot-message welcome-message">
                    <div class="message-avatar">🤖</div>
                    <div class="message-content">
                        <div class="message-text">Hello! I'm your AI assistant. I can help you with questions, brainstorming, or discussing your notes. What would you like to talk about?</div>
                        <div class="message-time">${this.getCurrentTime()}</div>
                    </div>
                </div>
            </div>
            <div class="chat-input-area">
                <div class="chat-input-container">
                    <textarea id="chat-input" placeholder="Type your message... (Cmd+/ to toggle chat)" autocomplete="off"></textarea>
                    <button id="chat-send-btn" type="button">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M2,21L23,12L2,3V10L17,12L2,14V21Z"/>
                        </svg>
                    </button>
                </div>
            </div>
            <div id="api-key-modal" class="api-key-modal" style="display: none;">
                <div class="api-key-modal-content">
                    <h4>OpenAI API Key</h4>
                    <p>Enter your OpenAI API key to enable AI responses:</p>
                    <input type="password" id="api-key-input" placeholder="sk-..." autocomplete="off">
                    <div class="api-key-buttons">
                        <button id="save-api-key">Save</button>
                        <button id="cancel-api-key">Cancel</button>
                        <button id="clear-api-key">Clear</button>
                    </div>
                    <small>Your API key is stored locally and never shared.</small>
                </div>
            </div>
        `;
    }

    initializeChat() {
        const chatInput = document.getElementById('chat-input');
        const sendBtn = document.getElementById('chat-send-btn');
        const chatMessages = document.getElementById('chat-messages');
        const apiKeyBtn = document.getElementById('api-key-btn');
        const clearChatBtn = document.getElementById('clear-chat-btn');
        const modelSelector = document.getElementById('model-selector');

        // Load API key and model from storage
        this.loadApiKey();
        this.loadSelectedModel();

        // Load chat history from storage
        this.loadChatHistory();

        const sendMessage = async () => {
            const message = chatInput.value.trim();
            if (message) {
                this.addChatMessage(message, 'user');
                chatInput.value = '';
                
                // Add user message to history
                this.chatHistory.push({ role: 'user', content: message });
                this.saveChatHistory();
                
                // Send to OpenAI or show fallback
                if (this.openaiApiKey) {
                    await this.sendToOpenAI();
                } else {
                    this.addChatMessage("Please configure your OpenAI API key to enable AI responses. Click the ⚙️ button in the chat header.", 'bot');
                }
            }
        };

        // Send button click
        sendBtn.addEventListener('click', sendMessage);

        // Enter key press
        chatInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
        });

        // Auto-resize textarea as content changes
        chatInput.addEventListener('input', () => {
            this.autoResizeTextarea(chatInput);
        });

        // Initial resize when chat opens
        setTimeout(() => {
            this.autoResizeTextarea(chatInput);
        }, 100);

        // API key button
        apiKeyBtn.addEventListener('click', () => {
            this.showApiKeyModal();
        });

        // Model selector change
        modelSelector.addEventListener('change', (e) => {
            this.selectedModel = e.target.value;
            this.saveSelectedModel();
            this.updateChatStatus();
        });

        // Clear chat button
        clearChatBtn.addEventListener('click', () => {
            if (confirm('Are you sure you want to clear all chat history?')) {
                this.clearChatHistory();
            }
        });

        // API key modal handlers
        this.setupApiKeyModal();

        // Focus input when chat opens
        setTimeout(() => chatInput.focus(), 100);
    }

    async loadApiKey() {
        try {
            const result = await chrome.storage.local.get(['openai-api-key']);
            this.openaiApiKey = result['openai-api-key'] || null;
            this.updateChatStatus();
        } catch (error) {
            console.error('Error loading API key:', error);
        }
    }

    async loadSelectedModel() {
        try {
            const result = await chrome.storage.local.get(['openai-selected-model']);
            this.selectedModel = result['openai-selected-model'] || 'gpt-4o-mini';
            
            // Update the dropdown selection
            const modelSelector = document.getElementById('model-selector');
            if (modelSelector) {
                modelSelector.value = this.selectedModel;
            }
            
            this.updateChatStatus();
        } catch (error) {
            console.error('Error loading selected model:', error);
        }
    }

    async saveSelectedModel() {
        try {
            await chrome.storage.local.set({ 'openai-selected-model': this.selectedModel });
        } catch (error) {
            console.error('Error saving selected model:', error);
        }
    }

    updateChatStatus() {
        const statusElement = document.getElementById('chat-status');
        if (statusElement) {
            if (this.openaiApiKey) {
                statusElement.textContent = `AI Ready (${this.getModelDisplayName()})`;
                statusElement.style.color = '#28a745';
            } else {
                statusElement.textContent = 'API key needed';
                statusElement.style.color = '#ffc107';
            }
        }
    }

    getModelDisplayName() {
        const modelNames = {
            'gpt-4o-mini': 'GPT-4o Mini',
            'gpt-4o': 'GPT-4o'
        };
        return modelNames[this.selectedModel] || this.selectedModel;
    }

    showApiKeyModal() {
        const modal = document.getElementById('api-key-modal');
        const input = document.getElementById('api-key-input');
        if (modal && input) {
            input.value = this.openaiApiKey || '';
            modal.style.display = 'flex';
            input.focus();
        }
    }

    setupApiKeyModal() {
        const modal = document.getElementById('api-key-modal');
        const input = document.getElementById('api-key-input');
        const saveBtn = document.getElementById('save-api-key');
        const cancelBtn = document.getElementById('cancel-api-key');
        const clearBtn = document.getElementById('clear-api-key');

        const closeModal = () => {
            modal.style.display = 'none';
        };

        saveBtn.addEventListener('click', async () => {
            const apiKey = input.value.trim();
            if (apiKey && apiKey.startsWith('sk-')) {
                this.openaiApiKey = apiKey;
                await chrome.storage.local.set({ 'openai-api-key': apiKey });
                this.updateChatStatus();
                closeModal();
                this.addChatMessage('API key saved! You can now chat with AI.', 'bot');
            } else {
                alert('Please enter a valid OpenAI API key (starts with sk-)');
            }
        });

        cancelBtn.addEventListener('click', closeModal);

        clearBtn.addEventListener('click', async () => {
            this.openaiApiKey = null;
            await chrome.storage.local.remove(['openai-api-key']);
            this.updateChatStatus();
            closeModal();
            this.addChatMessage('API key cleared. AI responses disabled.', 'bot');
        });

        // Close modal on outside click
        modal.addEventListener('click', (e) => {
            if (e.target === modal) {
                closeModal();
            }
        });

        // Save on Enter key
        input.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                saveBtn.click();
            }
        });
    }

    async sendToOpenAI() {
        const chatMessages = document.getElementById('chat-messages');
        const statusElement = document.getElementById('chat-status');
        
        // Create a message container for the streaming response
        const botMessageDiv = document.createElement('div');
        botMessageDiv.className = 'message bot-message';
        botMessageDiv.innerHTML = `
            <div class="message-avatar">🤖</div>
            <div class="message-content">
                <div class="message-text" id="streaming-response">
                    <div class="typing-indicator">●●●</div>
                </div>
                <div class="message-time">${this.getCurrentTime()}</div>
            </div>
        `;
        chatMessages.appendChild(botMessageDiv);
        chatMessages.scrollTop = chatMessages.scrollHeight;

        const streamingDiv = document.getElementById('streaming-response');
        let fullResponse = '';

        try {
            statusElement.textContent = 'AI is thinking...';
            
            const response = await fetch('https://api.openai.com/v1/chat/completions', {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${this.openaiApiKey}`,
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    model: this.selectedModel,
                    messages: this.chatHistory,
                    stream: true,
                    max_tokens: 500,
                    temperature: 0.7,
                }),
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const reader = response.body.getReader();
            const decoder = new TextDecoder();

            // Clear typing indicator
            streamingDiv.innerHTML = '';

            while (true) {
                const { done, value } = await reader.read();
                if (done) break;

                const chunk = decoder.decode(value);
                const lines = chunk.split('\n');

                for (const line of lines) {
                    if (line.startsWith('data: ')) {
                        const data = line.replace(/^data: /, '');
                        if (data === '[DONE]') {
                            break;
                        }
                        
                        try {
                            const parsed = JSON.parse(data);
                            const delta = parsed.choices?.[0]?.delta?.content || '';
                            if (delta) {
                                fullResponse += delta;
                                // Show plain text while streaming, render markdown when done
                                streamingDiv.textContent = fullResponse;
                                chatMessages.scrollTop = chatMessages.scrollHeight;
                            }
                        } catch (e) {
                            // Skip malformed JSON
                        }
                    }
                }
            }

            // Render final response as markdown
            streamingDiv.innerHTML = this.renderMarkdown(fullResponse);

            // Add data attributes to preserve raw content for persistence
            const messageDiv = streamingDiv.closest('.message');
            if (messageDiv) {
                messageDiv.setAttribute('data-raw-content', fullResponse);
                messageDiv.setAttribute('data-sender', 'bot');
            }

            // Add assistant response to history and save
            this.chatHistory.push({ role: 'assistant', content: fullResponse });
            this.saveChatHistory();
            this.saveChatMessages();
            statusElement.textContent = 'AI Ready';

        } catch (error) {
            console.error('OpenAI API error:', error);
            streamingDiv.innerHTML = `<span style="color: #dc3545;">Error: ${error.message}. Please check your API key and try again.</span>`;
            statusElement.textContent = 'Error occurred';
            // Save error message too
            this.saveChatMessages();
        }

        // Remove the streaming ID
        streamingDiv.removeAttribute('id');
    }

    addChatMessage(text, sender, shouldSave = true) {
        const chatMessages = document.getElementById('chat-messages');
        const messageDiv = document.createElement('div');
        messageDiv.className = `message ${sender}-message`;
        
        const avatar = sender === 'user' ? '👤' : '🤖';
        
        // Render markdown for bot messages using marked.js, plain text for user messages
        const messageContent = sender === 'bot' ? this.renderMarkdown(text) : this.escapeHtml(text);
        
        messageDiv.innerHTML = `
            <div class="message-avatar">${avatar}</div>
            <div class="message-content">
                <div class="message-text">${messageContent}</div>
                <div class="message-time">${this.getCurrentTime()}</div>
            </div>
        `;
        
        // Store the raw content in a data attribute for persistence
        messageDiv.setAttribute('data-raw-content', text);
        messageDiv.setAttribute('data-sender', sender);
        
        chatMessages.appendChild(messageDiv);
        chatMessages.scrollTop = chatMessages.scrollHeight;
        
        // Save to storage if requested
        if (shouldSave) {
            this.saveChatMessages();
        }
    }

    renderMarkdown(text) {
        try {
            // Check if libraries are available
            if (typeof marked === 'undefined') {
                console.error('marked.js is not loaded');
                return this.escapeHtml(text);
            }
            
            if (typeof DOMPurify === 'undefined') {
                console.error('DOMPurify is not loaded');
                return this.escapeHtml(text);
            }
            
            // Use marked.js to parse markdown to HTML
            const rawHtml = marked.parse(text, {
                gfm: true,           // Enable GitHub Flavored Markdown
                breaks: true,        // Convert \n to <br>
                sanitize: false,     // We'll handle sanitization separately
                smartLists: true,    // Use smarter list behavior
                smartypants: false   // Don't use smart quotes (can cause issues)
            });
            
            // Sanitize the HTML to prevent XSS attacks
            const sanitized = DOMPurify.sanitize(rawHtml, {
                ALLOWED_TAGS: [
                    'h1', 'h2', 'h3', 'h4', 'h5', 'h6',
                    'p', 'br', 'strong', 'b', 'em', 'i', 'u', 'del', 's',
                    'a', 'ul', 'ol', 'li', 'blockquote', 'code', 'pre',
                    'table', 'thead', 'tbody', 'tr', 'th', 'td',
                    'hr', 'img'
                ],
                ALLOWED_ATTR: [
                    'href', 'target', 'rel', 'src', 'alt', 'title',
                    'class', 'id'
                ],
                ALLOW_DATA_ATTR: false,
                ADD_ATTR: ['target'],
                ADD_TAGS: []
            });
            
            return sanitized;
            
        } catch (error) {
            console.error('Error in renderMarkdown:', error);
            return this.escapeHtml(text);
        }
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    getCurrentTime() {
        return new Date().toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
    }

    updateChatButton() {
        const chatBtn = document.getElementById('chat-toggle-status');
        if (chatBtn) {
            chatBtn.textContent = this.isChatOpen ? '✕ Close' : '💬 Chat';
            if (this.isChatOpen) {
                chatBtn.title = 'Close Chat (Cmd+/) - Highlight text to append to input';
            } else {
                chatBtn.title = 'Open Chat (Cmd+/) - Highlight text to add to chat';
            }
        }
    }

    async loadChatHistory() {
        try {
            const result = await chrome.storage.local.get(['newtab-chat-history', 'newtab-chat-messages']);
            
            // Load conversation history for API calls
            if (result['newtab-chat-history']) {
                this.chatHistory = result['newtab-chat-history'];
            } else {
                // Initialize with system message if no history exists
                this.chatHistory = [
                    {
                        role: 'system',
                        content: 'You are a helpful AI assistant integrated into a note-taking app. Be concise, friendly, and helpful. You can help with writing, brainstorming, questions, and discussing the user\'s notes.'
                    }
                ];
            }

            // Load and display chat messages in UI
            if (result['newtab-chat-messages']) {
                const chatMessages = document.getElementById('chat-messages');
                if (chatMessages) {
                    // Clear existing messages (except welcome message)
                    chatMessages.innerHTML = '';
                    
                    // Always start with welcome message
                    this.addWelcomeMessage();
                    
                    // Restore saved messages
                    const savedMessages = result['newtab-chat-messages'];
                    savedMessages.forEach(msg => {
                        this.addChatMessage(msg.content, msg.sender, false); // false = don't save to storage
                    });
                }
            }
        } catch (error) {
            console.error('Error loading chat history:', error);
            // Initialize with default if loading fails
            this.initializeDefaultChatHistory();
        }
    }

    async saveChatHistory() {
        try {
            await chrome.storage.local.set({ 
                'newtab-chat-history': this.chatHistory 
            });
        } catch (error) {
            console.error('Error saving chat history:', error);
        }
    }

    async saveChatMessages() {
        try {
            // Get all messages except the welcome message
            const chatMessages = document.getElementById('chat-messages');
            const messageElements = chatMessages.querySelectorAll('.message:not(.welcome-message)');
            
            const messages = Array.from(messageElements).map(el => {
                // Use data attributes to get raw content and sender
                const rawContent = el.getAttribute('data-raw-content');
                const sender = el.getAttribute('data-sender');
                
                // Fallback to old method if data attributes aren't available
                if (!rawContent || !sender) {
                    const isUser = el.classList.contains('user-message');
                    const textElement = el.querySelector('.message-text');
                    return {
                        content: textElement ? textElement.textContent : '',
                        sender: isUser ? 'user' : 'bot'
                    };
                }
                
                return {
                    content: rawContent,
                    sender: sender
                };
            });

            await chrome.storage.local.set({ 
                'newtab-chat-messages': messages 
            });
        } catch (error) {
            console.error('Error saving chat messages:', error);
        }
    }

    initializeDefaultChatHistory() {
        this.chatHistory = [
            {
                role: 'system',
                content: 'You are a helpful AI assistant integrated into a note-taking app. Be concise, friendly, and helpful. You can help with writing, brainstorming, questions, and discussing the user\'s notes.'
            }
        ];
    }

    addWelcomeMessage() {
        const chatMessages = document.getElementById('chat-messages');
        const welcomeDiv = document.createElement('div');
        welcomeDiv.className = 'message bot-message welcome-message';
        welcomeDiv.innerHTML = `
            <div class="message-avatar">🤖</div>
            <div class="message-content">
                <div class="message-text">Hello! I'm your AI assistant. I can help you with questions, brainstorming, or discussing your notes. What would you like to talk about?</div>
                <div class="message-time">${this.getCurrentTime()}</div>
            </div>
        `;
        chatMessages.appendChild(welcomeDiv);
    }

    async clearChatHistory() {
        try {
            // Reset conversation history
            this.chatHistory = [
                {
                    role: 'system',
                    content: 'You are a helpful AI assistant integrated into a note-taking app. Be concise, friendly, and helpful. You can help with writing, brainstorming, questions, and discussing the user\'s notes.'
                }
            ];

            // Clear UI messages
            const chatMessages = document.getElementById('chat-messages');
            if (chatMessages) {
                chatMessages.innerHTML = '';
                this.addWelcomeMessage();
            }

            // Clear storage
            await chrome.storage.local.remove(['newtab-chat-history', 'newtab-chat-messages']);
            
            // Show confirmation
            this.addChatMessage('Chat history cleared.', 'bot', false); // Don't save this message
            
        } catch (error) {
            console.error('Error clearing chat history:', error);
            this.addChatMessage('Error clearing chat history.', 'bot', false);
        }
    }
}

// Initialize the editor when the page loads
document.addEventListener('DOMContentLoaded', () => {
    // Wait for TOAST UI Editor to be available
    if (typeof toastui !== 'undefined') {
        new ToastUINotesEditor();
    } else {
        // Fallback: wait a bit more for the library to load
        setTimeout(() => {
            if (typeof toastui !== 'undefined') {
                new ToastUINotesEditor();
            } else {
                console.error('TOAST UI Editor failed to load');
            }
        }, 1000);
    }
}); 